{

    Unit AudioFileClass

    Defines and implements the class TAudioFile

    ---------------------------------------------------------------
    Covernator
    Copyright (C) 2010, Daniel Gaussmann
    http://www.gausi.de
    mail@gausi.de
    ---------------------------------------------------------------
    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA 02110, USA
    ---------------------------------------------------------------
}


unit AudioFileClass;

interface

uses windows, classes, SysUtils, Contnrs, Mp3FileUtils, ID3v2Frames;


type

    TAudioFile = class
    private
        // some properties
        // read it from ID3- or other tags

        fTitle    : String;
        FArtist   : String;
        FAlbum    : String;
        FGenre    : String;
        FYear     : String;
        fPath     : String;
        fComment  : String;

        fTrack: Byte;
        fDuration: Integer;

        function GetDirectory: String;
        function GetFilename: String;

        // Read tags from the filetype and convert the data to TAudiofile-Data
        procedure GetMp3Info(filename: UnicodeString; Flags: Integer = 0);

        // no tags found - set default values
        procedure SetUnknown;
    public

        property Title:     String Read fTitle write fTitle;
        property Artist:    String read fArtist write fArtist;
        property Album:     String read fAlbum write fAlbum;
        property Genre:     String read fGenre write fGenre;
        property Year:      String read fYear write fYear;
        property Comment:   String read fComment write fComment;

        property Directory: String read GetDirectory;
        property Filename: String read GetFilename;

        property Track: Byte read fTrack write fTrack;
        property Duration: Integer read fDuration write fDuration;

        property Path: String read fPath;


        constructor Create;
        destructor Destroy; override;

        procedure GetAudioData(filename: UnicodeString; Flags: Integer = 0);
        procedure Covernate(MimeTyp: AnsiString; Source: TStream);
        procedure ExCovernate;
    end;



const
      AUDIOFILE_UNKOWN = '';


procedure GetMp3Details(filename:UnicodeString;
      var mpeginfo:Tmpeginfo;
      var ID3v2Tag: TID3v2Tag;
      var id3v1tag:Tid3v1tag);


implementation

uses  CoverHelper;

 {$I-}


{
    --------------------------------------------------------
    GetMp3Details
      Read ID3v1Tag,
      Read ID3v2Tag,
      Read Mpeg-Info from file
    --------------------------------------------------------
}
procedure GetMp3Details(filename: UnicodeString;
    var mpeginfo:Tmpeginfo;
    var ID3v2Tag: TID3v2Tag;
    var id3v1tag:Tid3v1tag);
var
    Stream: TFileStream;
begin
    try
        Stream := TFileStream.Create(filename, fmOpenRead or fmShareDenyWrite);
        try
            id3v1tag.ReadFromStream(Stream);
            Stream.Seek(0, sobeginning);
            id3v2tag.ReadFromStream(Stream);
            if id3v2Tag.exists then
                Stream.Seek(id3v2tag.size, soFromBeginning)
            else
                Stream.Seek(0, sobeginning);
            mpeginfo.LoadFromStream(Stream);
        finally
            Stream.free;
        end;
    except
        // nothing here
        // if FileStream.Create throws an Exception just do nothing
        // this method is called too often. ;-)
    end;
end;


{
    --------------------------------------------------------
    Basic Class-Stuff.
    Create, Destroy, Assign
    --------------------------------------------------------
}
constructor TAudioFile.create;
begin
    inherited create;
    Duration := 0;
    Genre := 'Other';
    Year := '';
    Track := 0;
end;
destructor TAudioFile.Destroy;
begin

  inherited destroy;
end;


{
    --------------------------------------------------------
    Covernate
    Cover in ID3Tag einfgen
    --------------------------------------------------------
}
procedure TAudioFile.Covernate(MimeTyp: AnsiString;
  Source: TStream);
var Id3Tag: TID3v2Tag;
    PicFrames: TObjectList;
    i: Integer;
    dummystream: TMemoryStream;
    oldMime: AnsiString;
    oldPicType: Byte;
    oldDescription, newDescription: UnicodeString;
    existingFrontCover: TID3v2Frame;
begin
    Id3Tag := TID3v2Tag.Create;
    try
        ID3Tag.ReadFromFile(Path);
        if not Id3Tag.Exists then
        begin
            // set it (if values exists, they coe from the id3v1-Tag)
            Id3Tag.Artist := Artist;
            Id3Tag.Title := Title;
            Id3Tag.Album := Album;
            Id3Tag.Track := IntToStr(Track);
            Id3Tag.Year := Year;
            Id3Tag.Comment := Comment;
            Id3Tag.Genre := Genre;
        end;

        if (Artist <> '') and (album <> '') then
            newDescription := Artist + ' : ' + Album
        else
            newDescription := '';

        // Suche existierendes Front-Cover
        existingFrontCover := Nil;
        dummystream := TMemoryStream.Create;
        try
            PicFrames := Id3tag.GetAllPictureFrames;
            for i := 0 to PicFrames.Count - 1 do
            begin
                TID3v2Frame(PicFrames[i]).GetPicture(oldMime, oldPicType, oldDescription, dummystream);

                if oldPicType = 3 then // front Cover
                begin
                    existingFrontCover := TID3v2Frame(PicFrames[i]);
                    break;
                end;
            end;
        finally
              dummyStream.Free;
        end;

        if assigned(existingFrontCover) then
        begin
            existingFrontCover.SetPicture(MimeTyp, 3, newDescription, Source);
        end else
        begin
            // neues Front-Cover erstellen
            ID3Tag.SetPicture(MimeTyp, 3, newDescription, Source);
        end;

        ID3Tag.WriteToFile(Path);

    finally
        ID3Tag.Free;
    end;
end;

{
    --------------------------------------------------------
    ExCovernate
    Cover wieder entfernen
    --------------------------------------------------------
}
procedure TAudioFile.ExCovernate;
var Id3Tag: TID3v2Tag;
    PicFrames: TObjectList;
    i: Integer;
begin
    Id3Tag := TID3v2Tag.Create;
    try
        ID3Tag.ReadFromFile(Path);
        if Id3Tag.Exists then
        begin
            PicFrames := Id3tag.GetAllPictureFrames;
            for i := 0 to PicFrames.Count - 1 do
                ID3Tag.DeleteFrame(TID3v2Frame(PicFrames[i]));

            ID3Tag.WriteToFile(Path);
        end;
    finally
        ID3Tag.Free;
    end;
end;


{
    --------------------------------------------------------
    Some setter/getter
    --------------------------------------------------------
}
function TAudioFile.GetDirectory: String;
begin
    result := ExtractFilePath(Path);
end;

function TAudioFile.GetFilename: String;
begin
    result := ExtractFileName(Path);
end;

{
    --------------------------------------------------------
    GetAudioData, main method of this class
    --------------------------------------------------------
}
procedure TAudioFile.GetAudioData(filename: UnicodeString; Flags: Integer = 0);
begin
    try
        // Get the extension and call the proper private method.
        if (AnsiLowerCase(ExtractFileExt(filename)) = '.mp3')
            or (AnsiLowerCase(ExtractFileExt(filename)) = '.mp2')
            or (AnsiLowerCase(ExtractFileExt(filename)) = '.mp1')
        then
            GetMp3Info(Filename, Flags)
    except
        fPath := filename;
        SetUnknown;
    end;
end;

{
    --------------------------------------------------------
    GetMp3Info
    Uses MP3FileUtils
    --------------------------------------------------------
}
procedure TAudioFile.GetMp3Info(filename: UnicodeString; Flags: Integer = 0);
var mpegInfo:TMpegInfo;
    ID3v2Tag:TID3V2Tag;
    ID3v1tag:TID3v1Tag;
begin
    fPath := filename;
    if NOT FileExists(filename) then
    begin
      SetUnknown;
      exit;
    end;
    mpeginfo:=TMpegInfo.Create;
    ID3v2Tag:=TID3V2Tag.Create;
    ID3v1tag:=TID3v1Tag.Create;

    // read Tags from File
    GetMp3Details(filename,mpegInfo,ID3v2Tag,ID3v1tag);

    // If file was valid mp3-file:
    // Put the tag-info into the audiofile-structure.
    // Id3v2-tags have priority
    if mpeginfo.FirstHeaderPosition>-1 then
    begin
        // aus dem ID3 Tag
        if id3v2tag.artist<>'' then Artist:=(id3v2tag.artist)
            else if id3v1tag.artist<>'' then Artist:=(id3v1tag.artist)
                else Artist:=AUDIOFILE_UNKOWN;
        if id3v2tag.title<>'' then Title:=(id3v2tag.title)
            else if id3v1tag.title<>'' then Title:=(id3v1tag.title)
                else Title := Filename;
        if id3v2tag.album<>'' then Album:=(id3v2tag.album)
            else if id3v1tag.album<>'' then Album:=(id3v1tag.album)
                else album:=AUDIOFILE_UNKOWN;
        if id3v2tag.Year<>'' then Year:=(id3v2tag.Year)
            else if id3v1tag.Year<>'' then Year := Trim(UnicodeString(id3v1tag.Year))
                else Year:='';
        if id3v2tag.genre<>'' then
          genre := id3v2tag.genre
        else
          if id3v1tag.exists then
            genre := id3v1tag.genre
          else
            genre := '';
        Track := GetTrackFromV2TrackString(Id3v2tag.Track);
        if Track = 0 then
            Track := StrToIntDef(Id3v1tag.Track, 0);


        if id3v2tag.Comment<>'' then Comment := id3v2tag.Comment
          else if id3v1tag.Comment<>'' then Comment := id3v1tag.Comment
            else Comment := '';

        //  MPEG-Infos
        Duration:= mpeginfo.dauer;
    end else
    begin
        // if you're here:
        // the mp3-file is not valid. Use default-values.
        SetUnknown;
    end;
    MpegInfo.Free;
    Id3v2Tag.Free;
    Id3v1Tag.Free;
end;
{
    --------------------------------------------------------
    SetUnknown
    Default values, if no further information can be determined from the file
    --------------------------------------------------------
}
procedure TAudioFile.SetUnknown;
begin
  Title := Filename;
  Artist := AUDIOFILE_UNKOWN;
  Album := AUDIOFILE_UNKOWN;
  Year := '';
  Genre := '';
  Duration := 0;
  Track := 0;
end;


end.
